//============================================================================
// YKNR_SlideInformation.js - ver.1.0.0
// ---------------------------------------------------------------------------
// Copyright (c) 2017 Yakinori
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
//============================================================================
/*:
 * ===========================================================================
 * @plugindesc テキストを右からスライドインさせるお知らせウィンドウ
 * @author 焼きノリ
 * ===========================================================================
 *
 * @param 外部JSONファイル名
 * @desc 表示するテキストをJSONファイルから参照します。
 * 未指定なら読み込みません。詳細はヘルプをご確認ください。
 * @default 
 * @require 1
 * @dir data/
 * @type file
 *
 * @param ---------------------
 * @desc ----------------------------------------------------------
 * @default --------------------------------
 *
 * @param ウィンドウX座標
 * @desc 表示するウィンドウのX座標を設定します。
 * @default 340
 *
 * @param ウィンドウY座標
 * @desc 表示するウィンドウのY座標を設定します。
 * @default 0
 *
 * @param ウィンドウ幅
 * @desc ウィンドウを表示する幅を設定します。
 * 0 の場合は、X座標から画面端までのサイズが設定されます。
 * @default 0
 *
 * @param 表示行数
 * @desc 表示する行数を設定します。
 * @default 1
 *
 * @param ---------------------
 * @desc ----------------------------------------------------------
 * @default --------------------------------
 *
 * @param 一行辺りの高さ
 * @desc ウィンドウの高さを変更する場合はここで設定します。
 * 未指定なら、Window_BaseのlineHeightを参照します。
 * @default 30
 *
 * @param パディング
 * @desc ウィンドウのパディングを変更する場合はここで設定します。
 * 未指定なら、Window_BaseのstandardPaddingを参照します。
 * @default 10
 *
 * @param ウィンドウ透明度
 * @desc テキスト以外の透明度を変更する場合はここで設定します。
 * 未指定なら、デフォルトの 255 になります。
 * @default 255
 *
 * @param ウィンドウ背景透明度
 * @desc ウィンドウの背景の透明度を変更する場合はここで設定します。
 * 未指定なら、Window_BaseのstandardBackOpacityを参照します。
 * @default 192
 *
 * @param FontFace
 * @desc 使用するフォントを変更する場合はここで設定します。
 * 未指定なら、Window_BaseのstandardFontFaceを参照します。
 * @default GameFont
 *
 * @param フォントサイズ
 * @desc フォントサイズを変更する場合はここで設定します。
 * 未指定なら、Window_BaseのstandardFontSizeを参照します。
 * @default 22
 *
 * @param ウィンドウスキン
 * @desc 別のスキン使用する場合はここでファイル名を設定します。
 * 未指定なら、デフォルトの Window が設定されます。
 * @default Window
 * @require 1
 * @dir img/system/
 * @type file
 *
 * @param ---------------------
 * @desc ----------------------------------------------------------
 * @default --------------------------------
 *
 * @param 再生するSE_ファイル名
 * @desc ウィンドウが開かれたときに再生するSEの名前を設定します。
 * 未指定にすると再生を行いません。
 * @default Item1
 * @require 1
 * @dir audio/se/
 * @type file
 *
 * @param 再生するSE_音量
 * @desc ウィンドウが開かれたときに再生するSEの音量を設定します。
 * 未指定だと、デフォルトの 90 になります。
 * @default 90
 *
 * @param 再生するSE_ピッチ
 * @desc ウィンドウが開かれたときに再生するSEのピッチを設定します。
 * 未指定だと、デフォルトの 100 になります。
 * @default 100
 *
 * @param 再生するSE_位相
 * @desc ウィンドウが開かれたときに再生するSEの位相を設定します。
 * 未指定だと、デフォルトの 0 になります。
 * @default 0
 *
 * @param ---------------------
 * @desc ----------------------------------------------------------
 * @default --------------------------------
 *
 * @param スライドイン_時間
 * @desc スライドインする時間をフレーム単位で設定します。
 * 時間を指定する場合、速度は0に設定しないと動作しません。
 * @default 60
 *
 * @param スライドイン_速度
 * @desc スライドインする速度を設定します。
 * 速度を指定する場合、時間は0に設定しないと動作しません。
 * @default 0
 *
 * @param スライドイン_等速
 * @desc スライドインする動作を設定します。
 * 等速 : ON, カーブ : OFF
 * @default OFF
 *
 * @param スライドアウト_時間
 * @desc スライドアウトする時間をフレーム単位で設定します。
 * 時間を指定する場合、速度は0に設定しないと動作しません。
 * @default 0
 *
 * @param スライドアウト_速度
 * @desc スライドアウトする速度を設定します。
 * 速度を指定する場合、時間は0に設定しないと動作しません。
 * @default 2.5
 *
 * @param スライドインターバル
 * @desc スライドインとスライドアウトの間の一時停止する時間を
 * フレーム単位で指定します。
 * @default 150
 *
 * @help
 * ===========================================================================
 * テキストを右からスライドインさせるお知らせウィンドウを提供します。
 * デフォルトでは画面上部に、マップ表示と被らないように表示しています。
 * ===========================================================================
 *【機能紹介】
 * 次の目的地や何かお知らせを画面に簡易的に表示するとき、
 * イベントのメッセージを出すことでユーザーに親切な設計にすることができますが
 * それだと、メッセージが終わるまでプレイヤーを動かすことができません。
 *
 * 本プラグインでは、そのような簡易なお知らせに
 * ぴったりなウィンドウを表示できるようになります。
 *
 *
 *【基本仕様】
 * ・プラグインパラメータにより、画面の好きな位置、幅で、
 * 　右から左へテキストをスライドさせて表示できる
 * ・マップ上でもバトル中でも表示可能
 * ・プラグインパラメータ「外部JSONファイル名」でJSONファイルを指定すると、
 * 　そのJSONファイルからテキストを呼び出すことができるようになる（後述）
 *
 *
 *【ゲーム中の仕様】
 * ・プレイヤーの進行を妨げず、お知らせ表示中でもキャラを動かすことができる
 * ・お知らせウィンドウが機能するシーンは、マップ、バトルです
 * ・お知らせウィンドウ動作中に以下の状態になると、
 * 　テキストが全部表示されていなくても強制的に閉じます
 * 　・イベントの実行
 * 　・メニューを開く
 * 　・バトルが始まる
 * 　・バトルから戻る
 * ・マップ名が表示されている最中にお知らせを表示すると、
 * 　表示位置によってはマップ名に重なって表示されます。
 *
 *
 *【プラグインコマンド】
 * 3種類のコマンドを用意しています。
 *
 * ---------------------------------------------------------------------------
 * お知らせの表示 急いで\C[1]王様\C[0]の所へ戻ろう！
 * ---------------------------------------------------------------------------
 * 表示するテキストを直接記述するタイプです。制御文字も使用可能です。
 *
 * ---------------------------------------------------------------------------
 * お知らせの表示 $ID:王が呼んでる
 * ---------------------------------------------------------------------------
 * 外部JSONファイルにあるIDを指定し、そのテキストを表示します。
 * $ と : は必ず半角にしてください。
 *
 * ---------------------------------------------------------------------------
 * お知らせ再表示
 * ---------------------------------------------------------------------------
 * 前回表示したお知らせをもう一度表示します。
 *
 *
 *【外部JSONファイル名について】
 * プラグインパラメータの「外部JSONファイル名」の設定により、
 * 別途用意されたJSONファイルからテキストを参照して表示することができます。
 * 利用する場合は、以下の3点を確認してください。
 *
 * ・パラメータのファイル名には拡張子(.json)まで記述すること
 * ・用意するJSONファイルは、プロジェクトフォルダのdataフォルダ内に置くこと
 * ・JSONファイルのデータの構成は以下の例を参考にすること
 * ---------------------------------------------------------------------------
 * {
 *     "ここにID" : "ここに表示するテキスト",
 *     "王が呼んでる" : "急いで\\C[1]王様\\C[0]の所へ戻ろう！",
 *     "test" : "お知らせの外部メッセージ呼び出しテストです。"
 * }
 * ---------------------------------------------------------------------------
 *
 *
 *【スライドパラメータ設定例】
 *
 * ■デフォルトで設定している値
 * 　　スライドイン_時間：60
 * 　　スライドイン_速度：0
 * 　　スライドイン_等速：OFF
 * 　　スライドアウト_時間：0
 * 　　スライドアウト_速度：2.5
 * 　　スライドインターバル：150
 *
 * ■右から左へ等速で移動、途中は停止せずに最後まで等速。
 * 　全部表示したら閉じる
 * 　　スライドイン_時間：0
 * 　　スライドイン_速度：3
 * 　　スライドイン_等速：ON
 * 　　スライドアウト_時間：0
 * 　　スライドアウト_速度：3
 * 　　スライドインターバル：0
 *
 * ■スライドせずにパッと表示、150F停止後、スライドせずに閉じる。
 * 　ウィンドウより長いテキストは見切れる
 * 　　スライドイン_時間：0
 * 　　スライドイン_速度：0
 * 　　スライドイン_等速：ON
 * 　　スライドアウト_時間：0
 * 　　スライドアウト_速度：0
 * 　　スライドインターバル：150
 *
 *
 * ---------------------------------------------------------------------------
 *【更新履歴】
 * [2017/03/05] [1.0.0] 公開
 *
 * ===========================================================================
 * [Blog]   : http://mata-tuku.ldblog.jp/
 * [Twitter]: https://twitter.com/Noritake0424
 * ---------------------------------------------------------------------------
 * 本プラグインは MITライセンス のもとで配布されています。
 * 利用はどうぞご自由に。
 * http://opensource.org/licenses/mit-license.php
*/

(function () {
    'use strict';

    //------------------------------------------------------------------------
    // パラメータを受け取る.
    var parameters = PluginManager.parameters('YKNR_SlideInformation');

    var messageDatabase = parameters['外部JSONファイル名'];

    var windowX = parseInt(parameters['ウィンドウX座標']) || 0;
    var windowY = parseInt(parameters['ウィンドウY座標']) || 0;
    var windowW = parseInt(parameters['ウィンドウ幅']) || 0;
    var lineNum = parseInt(parameters['表示行数']) || 1;

    var lineHeight = parseInt(parameters['一行辺りの高さ']) || 0;
    var padding = parseInt(parameters['パディング']) || -1;
    var opacity = parseInt(parameters['透明度']) || -1;
    var backOpacity = parseInt(parameters['背景透明度']) || -1;
    var fontFace = parameters['FontFace'];
    var fontSize = parseInt(parameters['フォントサイズ']) || 0;
    var windowskin = parameters['ウィンドウスキン'];

    var seName = parameters['再生するSE_ファイル名'];
    var seVolume = parseInt(parameters['再生するSE_音量']) || 90;
    var sePitch = parseInt(parameters['再生するSE_ピッチ']) || 100;
    var sePan = parseInt(parameters['再生するSE_位相']) || 0;

    var slideInDuration = parseInt(parameters['スライドイン_時間']);
    var slideInSpeed = Number(parameters['スライドイン_速度']);
    var slideInLiner = parameters['スライドイン_等速'].toUpperCase() === 'ON';
    var slideOutDuration = parseInt(parameters['スライドアウト_時間']);
    var slideOutSpeed = Number(parameters['スライドアウト_速度']);
    var slideInterval = parseInt(parameters['スライドインターバル']);

    //------------------------------------------------------------------------
    // Window_Information

    function Window_Information() {
        this.initialize.apply(this, arguments);
    }

    Window_Information.prototype = Object.create(Window_Base.prototype);
    Window_Information.prototype.constructor = Window_Information;

    Window_Information.prototype.initialize = function () {
        var w = (windowW > 0) ? windowW : Graphics.boxWidth - windowX;
        var h = this.fittingHeight(lineNum);
        Window_Base.prototype.initialize.call(this, windowX, windowY, w, h);
        this.hide();
        this.openness = 0;
        if (opacity >= 0) {
            this.opacity = opacity;
        }
        this.initMembers();
    };

    Window_Information.prototype.initMembers = function () {
        this._text = '';
        this._textState = null;
        this._endClose = false;
        this._isRunningCurrentEvent = false;
        this._isObserveEvent = false;
    };

    Window_Information.prototype.lineHeight = function () {
        if (lineHeight > 0) {
            return lineHeight;
        } else {
            return Window_Base.prototype.lineHeight.call(this);
        }
    };

    Window_Information.prototype.standardFontFace = function () {
        if (fontFace) {
            return fontFace;
        } else {
            return Window_Base.prototype.standardFontFace.call(this);
        }
    };

    Window_Information.prototype.standardFontSize = function () {
        if (fontSize > 0) {
            return fontSize;
        } else {
            return Window_Base.prototype.standardFontSize.call(this);
        }
    };

    Window_Information.prototype.standardPadding = function () {
        if (padding >= 0) {
            return padding;
        } else {
            return Window_Base.prototype.standardPadding.call(this);
        }
    };

    Window_Information.prototype.standardBackOpacity = function () {
        if (backOpacity >= 0) {
            return backOpacity;
        } else {
            return Window_Base.prototype.standardBackOpacity.call(this);
        }
    };

    Window_Information.prototype.loadWindowskin = function () {
        if (windowskin) {
            this.windowskin = ImageManager.loadSystem(windowskin);
        } else {
            Window_Base.prototype.loadWindowskin.call(this);
        }
    };

    Window_Information.prototype.close = function () {
        Window_Base.prototype.close.call(this);
        this.terminateMessage();
    };

    Window_Information.prototype.terminateMessage = function () {
        if (this._endClose) {
            $gameTemp.closeInformation();
        }
        this._textState = null;
        this._isRunningCurrentEvent = false;
        this._isObserveEvent = false;
    };

    Window_Information.prototype.update = function () {
        Window_Base.prototype.update.call(this);
        this.updateInformation();
    };

    Window_Information.prototype.updateClose = function () {
        if (this.isClosing()) {
            Window_Base.prototype.updateClose.call(this);
            if (this.isClosed()) {
                this.hide();
            }
        }
    };

    Window_Information.prototype.updateInformation = function () {
        while (!this.isOpening() && !this.isClosing()) {
            this.updateObserveEventRunning();
            if (this.updateMessage()) {
                return;
            } else if (this.canStart()) {
                this.startMessage();
            } else if (this.isAutoEndMessage()) {
                this._endClose = false;
                this.endMessage();
                return;
            } else {
                return;
            }
        }
    };

    Window_Information.prototype.updateObserveEventRunning = function () {
        if (this._isObserveEvent) {
            if ($gameMap.isEventRunning() || $gameTroop.isEventRunning()) {
                this._isRunningCurrentEvent = true;
            } else {
                this._isRunningCurrentEvent = false;
                this._isObserveEvent = false;
            }
        }
    };

    Window_Information.prototype.canStart = function () {
        if (!$gameMap.isEventRunning() && !$gameTroop.isEventRunning()) {
            return $gameTemp.canStartInformation() && !!$gameTemp.getInformationMessage();
        }
        return false;
    };

    Window_Information.prototype.startMessage = function () {
        this._isObserveEvent = true;
        this._text = $gameTemp.getInformationMessage();
        var convert_text = this.convertEscapeCharacters(this._text);
        this._textState = {};
        this._textState.phese = 0;
        this._textState.phese1_counter = 0;
        this._textState.width = this.textWidth(convert_text);
        this._textState.width += this.standardFontSize();
        this._textState.slidein = this.createTweenObject();
        this._textState.slidein.from = -this.contentsWidth();
        this._textState.slidein.to = 0;
        this._textState.slidein.speed = slideInSpeed;
        this._textState.slidein.duration = slideInDuration;
        if (!slideInLiner) {
            this._textState.slidein.easing = this.updateSlideCurve;
        }
        this._textState.slideout = this.createTweenObject();
        this._textState.slideout.from = 0;
        this._textState.slideout.to = this._textState.width;
        this._textState.slideout.speed = slideOutSpeed;
        this._textState.slideout.duration = slideOutDuration;

        this.origin.x = this._textState.slidein.calc();

        if (seName) {
            var playSeParams = {
                name: seName,
                volume: seVolume,
                pitch: sePitch,
                pan: sePan
            };
            AudioManager.playSe(playSeParams);
        }

        this.refresh();
        this.show();
        this.open();
    };

    Window_Information.prototype.endMessage = function () {
        this.close();
        this._endClose = false;
    };

    Window_Information.prototype.updateMessage = function () {
        if (this.isRunningMessage()) {
            switch (this._textState.phese) {
                case 0:
                    this.origin.x = this._textState.slidein.calc();
                    if (this._textState.slidein.progress < 1) {
                        return true;
                    }
                    this._textState.phese++;
                case 1:
                    if (++this._textState.phese1_counter < slideInterval) {
                        return true;
                    }
                    this._textState.phese++;
                case 2:
                    this.origin.x = this._textState.slideout.calc();
                    if (this._textState.slideout.progress < 1) {
                        return true;
                    }
                    this._textState.phese++;
            }
            this._endClose = true;
            this.endMessage();
            return true;
        } else {
            return false;
        }
    };

    Window_Information.prototype.createTweenObject = function () {
        return {
            _rate: function (current) {
                var rate = 1;
                if (this.duration > 0 && this.speed === 0) {
                    rate = current / this.duration;
                } else if (this.duration === 0 && this.speed > 0) {
                    var total = (this.to - this.from) / this.speed;
                    rate = current / total;
                }
                return rate.clamp(0, 1);
            },
            calc: function () {
                this.progress = this._rate(this._current++);
                var easing_rate = this.easing(this.progress);
                return this.from + (this.to - this.from) * easing_rate;
            },
            easing: function (rate) { return rate; },
            _current: 0,
            progress: 0,
            from: 0,
            to: 0,
            duration: 0,
            speed: 0,
        };
    };

    Window_Information.prototype.updateSlideCurve = function (k) {
        // Tween Cubic Out.
        return --k * k * k + 1;
    };

    Window_Information.prototype.isRunningMessage = function () {
        if (this._textState) {
            return !this.isAutoEndMessage();
        }
        return false;
    };

    Window_Information.prototype.isAutoEndMessage = function () {
        if (this.isOpen() && !this._isRunningCurrentEvent) {
            if ($gameMap.isEventRunning() || $gameTroop.isEventRunning()) {
                return true;
            }
            return this._text != $gameTemp.getInformationMessage();
        }
        return false;
    };

    Window_Information.prototype.resizeContents = function (width, height) {
        this.contents.resize(width, height);
        this._windowContentsSprite._refresh();
    };

    Window_Information.prototype.refresh = function () {
        var w = this._textState ? this._textState.width : this.contents.width;
        var h = this.contents.height;
        this.contents.clear();
        this.resizeContents(w, h);
        this.drawTextEx(this._text, this.textPadding(), 0);
    };

    //------------------------------------------------------------------------
    DataManager.loadInformationData = function (filename) {
        var objectname = '$SlideInformationData';
        this.loadDataFile(objectname, filename);
    };

    DataManager.isInformationLoaded = function () {
        this.checkError();
        return !!$SlideInformationData;
    };

    //------------------------------------------------------------------------
    Game_Temp.prototype.getInformationMessage = function () {
        if (this._informationMessage === undefined) {
            this._informationMessage = '';
        }
        return this._informationMessage;
    };

    Game_Temp.prototype.setInformationMessage = function (message) {
        this._informationMessage = message;
    };

    Game_Temp.prototype.setInformationMessageId = function (id) {
        this.setInformationMessage($SlideInformationData[id]);
    };

    Game_Temp.prototype.canStartInformation = function () {
        return !!this._openInformation;
    };

    Game_Temp.prototype.openInformation = function () {
        this._openInformation = !!this._informationMessage;
    };
    Game_Temp.prototype.closeInformation = function () {
        this._openInformation = false;
    };

    //------------------------------------------------------------------------
    var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function (command, args) {
        if (command === 'お知らせの表示') {
            if (args[0].length > 4 && args[0].substr(0, 4).toUpperCase() === '$ID:') {
                $gameTemp.setInformationMessageId(args[0].substr(4));
            } else {
                $gameTemp.setInformationMessage(args[0]);
            }
            $gameTemp.openInformation();
        } else if (command === 'お知らせ再表示') {
            $gameTemp.openInformation();
        } else {

            _Game_Interpreter_pluginCommand.call(this, command, args);
        }
    };

    //------------------------------------------------------------------------
    if (messageDatabase) {
        var _Scene_Boot_create = Scene_Boot.prototype.create;
        Scene_Boot.prototype.create = function () {
            _Scene_Boot_create.call(this);
            DataManager.loadInformationData(messageDatabase);
        };

        var _Scene_Boot_isReady = Scene_Boot.prototype.isReady;
        Scene_Boot.prototype.isReady = function () {
            if (_Scene_Boot_isReady.call(this)) {
                return DataManager.isInformationLoaded();
            } else {
                return false;
            }
        };
    }
    //------------------------------------------------------------------------
    var _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function () {
        _Scene_Map_createAllWindows.call(this);
        this.createInformationWindow();
    };

    Scene_Map.prototype.createInformationWindow = function () {
        this._informationWindow = new Window_Information();
        this.addChild(this._informationWindow);
        // マップ以外のシーンから来た場合はメッセージをリセット.
        // マップでお知らせ表示中に
        if (!SceneManager.isPreviousScene(Scene_Map)) {
            $gameTemp.setInformationMessage('');
        }
    };

    //------------------------------------------------------------------------
    var _Scene_Battle_terminate = Scene_Battle.prototype.terminate;
    Scene_Battle.prototype.terminate = function () {
        _Scene_Battle_terminate.call(this);
        $gameTemp.setInformationMessage('');
    };

    var _Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
    Scene_Battle.prototype.createAllWindows = function () {
        _Scene_Battle_createAllWindows.call(this);
        this.createInformationWindow();
    };

    Scene_Battle.prototype.createInformationWindow = function () {
        this._informationWindow = new Window_Information();
        this.addWindow(this._informationWindow);
        $gameTemp.setInformationMessage('');
    };

})();
